/////////////////////////////////////////////////////////////
// CINEMA 4D SDK                                           //
/////////////////////////////////////////////////////////////
// (c) 1989-2006 MAXON Computer GmbH, all rights reserved  //
/////////////////////////////////////////////////////////////

// this example demonstrates how to implement a simple value driven type effector

#include "c4d.h"
#include "c4d_symbols.h"
#include "c4d_baseeffectorplugin.h"
#include "oenoise.h"

class NoiseEffector : public EffectorData
{
public:
		Real size;

		virtual Bool		InitEffector(GeListNode *node);

		virtual void		InitPoints(BaseObject *op,BaseObject *gen,BaseDocument *doc,EffectorDataStruct *data,MoData *md,BaseThread *thread);

		virtual void		CalcPointValue(BaseObject	*op, BaseObject	*gen,	BaseDocument *doc, EffectorDataStruct *data, LONG index, MoData	*md, const Vector	&globalpos, Real fall_weight);
		virtual Vector	CalcPointColor(BaseObject	*op, BaseObject	*gen, BaseDocument *doc, EffectorDataStruct *data, LONG index, MoData	*md, const Vector	&globalpos, Real fall_weight);

		static NodeData* Alloc(void) { return gNew NoiseEffector; }
};

//Called when effector is first created
Bool NoiseEffector::InitEffector(GeListNode *node)
{
	BaseObject *op=(BaseObject*)node;
	if (!op) return FALSE;

	BaseContainer *bc=op->GetDataInstance();
	if (!bc) return FALSE;

	bc->SetReal(ID_MG_BASEEFFECTOR_MINSTRENGTH, -1.0);
	bc->SetBool(ID_MG_BASEEFFECTOR_POSITION_ACTIVE, TRUE);
	bc->SetVector(ID_MG_BASEEFFECTOR_POSITION, 50.0);
	bc->SetReal(NOISEEFFECTOR_SCALE, 0.01);

	return TRUE;
}

//Called just before points calculation
void NoiseEffector::InitPoints(BaseObject *op,BaseObject *gen,BaseDocument *doc,EffectorDataStruct *data,MoData *md,BaseThread *thread)
{
	BaseContainer *bc=op->GetDataInstance();
	if (!bc) return;

	size=bc->GetReal(NOISEEFFECTOR_SCALE);
}

//This example shows two methods to setting the values that drive the effector
//change this value to 1 to try the second method
const LONG EffectorMethod = 0;

void NoiseEffector::CalcPointValue(BaseObject	*op, BaseObject	*gen,	BaseDocument *doc, EffectorDataStruct *data, LONG index, MoData	*md, const Vector	&globalpos, Real fall_weight)
{
	switch (EffectorMethod)
	{
		case 0:
			{	
				//First Method, this one just iterates through the "blends" which are the raw driver values and is slightly faster
				LONG i=0;
				Real *buf=data->strengths;
				for (i=0;i<BLEND_COUNT;i++,buf++)
				{
					(*buf) = Noise( ( globalpos + (i * 20.0) ) * size );
				}
			} break;
		case 1:
			{
				//Method 2, this casts the the more user friendly EffectorStrengths structure for setting of values
				EffectorStrengths *es=(EffectorStrengths*)data->strengths;
				//Position
				es->pos = Vector( Noise( globalpos * size), 
													Noise( ( globalpos + Vector( 10.0 ) ) * size ), 
													Noise( ( globalpos - Vector( 10.0 ) ) * size ) );
				//Rotation
				es->rot = Vector( Noise( ( globalpos + Vector(20.0, 60.0, -90.0) ) * size ), 
													Noise( ( globalpos + Vector(-20.0, 60.0, 90.0) ) * size ),  
													Noise( ( globalpos + Vector(20.0, -60.0, 90.0) ) * size ) );
				//Scale
				es->scale = Vector( Noise( ( globalpos + Vector(120.0, -160.0, -190.0) ) * size ), 
														Noise( ( globalpos + Vector(100.0, 160.0, 190.0) ) * size ),  
														Noise( ( globalpos + Vector(160.0, 160.0, 190.0) ) * size ) );
				//Color opacity
				es->col = Vector( Noise( globalpos * size ) );
				//Others, x = U, y = V, z = Visibility
				es->other = Vector( Noise( ( globalpos + Vector(40.0, 70.0, -10.0) ) * size ), 
														Noise( ( globalpos + Vector(-40.0, 70.0, 10.0) ) * size ),  
														Noise( ( globalpos + Vector(40.0, -70.0, 10.0) ) * size ) );
				//Others2, x = weight, y = clone index, z = time
				es->other2 = Vector( Noise( ( globalpos + Vector(90.0, 20.0, -50.0) ) * size ), 
														Noise( ( globalpos + Vector(-90.0, 20.0, 50.0) ) * size ),  
														Noise( ( globalpos + Vector(90.0, -20.0, 50.0) ) * size ) );
				//Others3 <reserved>
				es->other3 = Vector( Noise( ( globalpos + Vector(90.0, 20.0, -50.0) ) * size ), 
														Noise( ( globalpos + Vector(-90.0, 20.0, 50.0) ) * size ),  
														Noise( ( globalpos + Vector(90.0, -20.0, 50.0) ) * size ) );
			} break;
		default: break;
	}
}

//Calculcate the effector color
Vector NoiseEffector::CalcPointColor(BaseObject	*op, BaseObject	*gen, BaseDocument *doc, EffectorDataStruct *data, LONG index, MoData	*md, const Vector	&globalpos, Real fall_weight)
{
	return Vector(Noise(globalpos),Noise(globalpos+20),Noise(globalpos-20));
}

// be sure to use a unique ID obtained from www.plugincafe.com
#define ID_NOISEEFFECTOR 1019570

Bool RegisterNoiseEffector(void)
{
	return RegisterEffectorPlugin(ID_NOISEEFFECTOR,GeLoadString(IDS_NOISEEFFECTOR),OBJECT_CALL_ADDEXECUTION,NoiseEffector::Alloc,"oenoise",AutoBitmap("noiseeffector.tif"),0);
}
